<?php
header("Content-Type: application/json");
require_once 'db_connect.php';

// Increase limits for uploads
ini_set('max_execution_time', 300);
ini_set('upload_max_filesize', '20M');
ini_set('post_max_size', '25M');
ini_set('memory_limit', '128M');

$response = ['success' => false, 'message' => ''];

// Log upload attempt
error_log('Upload attempt - Method: ' . $_SERVER['REQUEST_METHOD']);

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    $response['message'] = 'Invalid request method';
    echo json_encode($response);
    exit;
}

// Check if uploads directory exists and is writable
$uploadDir = 'uploads/';
if (!is_dir($uploadDir)) {
    if (!mkdir($uploadDir, 0755, true)) {
        $response['message'] = 'Failed to create upload directory';
        echo json_encode($response);
        exit;
    }
}

if (!is_writable($uploadDir)) {
    $response['message'] = 'Upload directory is not writable';
    echo json_encode($response);
    exit;
}

// Validate parameters
$userId = filter_input(INPUT_POST, 'user_id', FILTER_VALIDATE_INT);
$mediaType = in_array($_POST['media_type'] ?? null, ['image', 'video']) ? $_POST['media_type'] : null;
$gradientIndex = filter_input(INPUT_POST, 'gradient_index', FILTER_VALIDATE_INT);

error_log("Params - UserID: $userId, Type: $mediaType, Gradient: $gradientIndex");

if (!$userId || !$mediaType || !isset($_FILES['media']) || $gradientIndex === false) {
    $response['message'] = 'Missing or invalid parameters';
    echo json_encode($response);
    exit;
}

// File validation
$allowedMimeTypes = [
    'image' => ['image/jpeg', 'image/png', 'image/gif', 'image/webp'],
    'video' => ['video/mp4', 'video/quicktime', 'video/x-msvideo', 'video/x-matroska']
];

$allowedMusicTypes = [
    'audio/mpeg', 
    'audio/mp3', 
    'audio/mp4',
    'audio/x-m4a',
    'audio/aac',
    'audio/wav',
    'audio/x-wav',
    'audio/ogg',
    'application/octet-stream' // Fallback for unknown types
];

$maxSize = 20 * 1024 * 1024; // 20MB

$file = $_FILES['media'];
error_log("Media file - Name: {$file['name']}, Type: {$file['type']}, Size: {$file['size']}, Error: {$file['error']}");

// Check file upload error
if ($file['error'] !== UPLOAD_ERR_OK) {
    $errorMessages = [
        0 => 'There is no error, the file uploaded with success',
        1 => 'The uploaded file exceeds the upload_max_filesize directive in php.ini',
        2 => 'The uploaded file exceeds the MAX_FILE_SIZE directive that was specified in the HTML form',
        3 => 'The uploaded file was only partially uploaded',
        4 => 'No file was uploaded',
        6 => 'Missing a temporary folder',
        7 => 'Failed to write file to disk.',
        8 => 'A PHP extension stopped the file upload.'
    ];
    $response['message'] = 'File upload error: ' . ($errorMessages[$file['error']] ?? 'Unknown error');
    echo json_encode($response);
    exit;
}

// Validate media type and size
if (!isset($allowedMimeTypes[$mediaType]) || 
    !in_array($file['type'], $allowedMimeTypes[$mediaType]) || 
    $file['size'] > $maxSize) {
    $response['message'] = 'Invalid media file. Type: ' . $file['type'] . ', Size: ' . $file['size'];
    echo json_encode($response);
    exit;
}

// Handle music file
$musicPath = null;
if (isset($_FILES['music']) && $_FILES['music']['error'] === UPLOAD_ERR_OK) {
    $musicFile = $_FILES['music'];
    error_log("Music file - Name: {$musicFile['name']}, Type: {$musicFile['type']}, Size: {$musicFile['size']}");
    
    if (in_array($musicFile['type'], $allowedMusicTypes) && $musicFile['size'] <= $maxSize) {
        $musicExtension = pathinfo($musicFile['name'], PATHINFO_EXTENSION);
        if (empty($musicExtension)) {
            // Try to determine extension from MIME type
            $mimeToExt = [
                'audio/mpeg' => 'mp3',
                'audio/mp3' => 'mp3',
                'audio/mp4' => 'm4a',
                'audio/x-m4a' => 'm4a',
                'audio/aac' => 'aac',
                'audio/wav' => 'wav',
                'audio/x-wav' => 'wav',
                'audio/ogg' => 'ogg'
            ];
            $musicExtension = $mimeToExt[$musicFile['type']] ?? 'mp3';
        }
        
        $musicFileName = uniqid() . '_music.' . $musicExtension;
        $musicPath = $uploadDir . $musicFileName;
        
        if (move_uploaded_file($musicFile['tmp_name'], $musicPath)) {
            error_log("Music file uploaded successfully: $musicPath");
        } else {
            error_log("Failed to move music file");
            $musicPath = null;
        }
    } else {
        error_log("Invalid music file: Type={$musicFile['type']}, Size={$musicFile['size']}");
    }
} else if (isset($_FILES['music'])) {
    error_log("Music file upload error: " . $_FILES['music']['error']);
}

// Generate unique filename for media
$extension = pathinfo($file['name'], PATHINFO_EXTENSION);
if (empty($extension)) {
    // Determine extension from MIME type
    $mimeToExt = [
        'image/jpeg' => 'jpg',
        'image/png' => 'png',
        'image/gif' => 'gif',
        'image/webp' => 'webp',
        'video/mp4' => 'mp4',
        'video/quicktime' => 'mov',
        'video/x-msvideo' => 'avi',
        'video/x-matroska' => 'mkv'
    ];
    $extension = $mimeToExt[$file['type']] ?? 'bin';
}

$fileName = uniqid() . '_' . time() . '.' . $extension;
$uploadPath = $uploadDir . $fileName;

// Move uploaded file
if (move_uploaded_file($file['tmp_name'], $uploadPath)) {
    error_log("Media file moved to: $uploadPath");
    
    try {
        $expiresAt = date('Y-m-d H:i:s', strtotime('+24 hours'));
        $stmt = $pdo->prepare("INSERT INTO stories 
                              (user_id, media_url, media_type, music_url, gradient_index, expires_at) 
                              VALUES (?, ?, ?, ?, ?, ?)");
        
        if ($stmt->execute([$userId, $uploadPath, $mediaType, $musicPath, $gradientIndex, $expiresAt])) {
            $response = [
                'success' => true,
                'message' => 'Story uploaded successfully',
                'url' => $uploadPath,
                'music_url' => $musicPath,
                'gradient_index' => $gradientIndex,
                'expires_at' => $expiresAt
            ];
            error_log("Story saved to database");
        } else {
            $response['message'] = 'Failed to save to database';
            error_log("Database save failed");
        }
    } catch (PDOException $e) {
        // Clean up files on error
        if (file_exists($uploadPath)) unlink($uploadPath);
        if ($musicPath && file_exists($musicPath)) unlink($musicPath);
        
        $response['message'] = 'Database error';
        error_log('Story upload database error: ' . $e->getMessage());
    }
} else {
    $response['message'] = 'Failed to move uploaded file';
    error_log("Failed to move file to: $uploadPath");
}

echo json_encode($response);
?>