<?php 
error_reporting(E_ALL);
ini_set('display_errors', 1);
require 'vendor/autoload.php';

use Google\Auth\OAuth2;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;

// ----------------------- FUNCTION TO SEND NOTIFICATION -----------------------
function sendNotificationForMessage($messageId) {
    $serviceAccountPath = __DIR__ . '/service-account.json';
    $projectId = 'dreams-9559e';

    try {
        $pdo = new PDO("mysql:host=localhost;dbname=dreams", "root", "");
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    } catch (PDOException $e) {
        error_log("Database connection failed: " . $e->getMessage());
        return false;
    }

    $stmt = $pdo->prepare("
        SELECT m.*, 
               sender.username AS sender_username,
               sender.reels_video AS sender_reels_video,
               receiver.fcm_token AS receiver_fcm_token
        FROM messages m
        LEFT JOIN login sender ON m.sender_id = sender.id
        LEFT JOIN login receiver ON m.receiver_id = receiver.id
        WHERE m.id = ?
    ");
    
    if (!$stmt->execute([$messageId])) {
        error_log("Error executing query: " . implode(" ", $stmt->errorInfo()));
        return false;
    }

    $messageRow = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$messageRow || empty($messageRow['receiver_fcm_token'])) return false;

    $credentials = json_decode(file_get_contents($serviceAccountPath), true);
    if (json_last_error() !== JSON_ERROR_NONE) return false;

    try {
        $oauth = new OAuth2([
            'audience' => 'https://oauth2.googleapis.com/token',
            'issuer' => $credentials['client_email'],
            'signingAlgorithm' => 'RS256',
            'signingKey' => $credentials['private_key'],
            'tokenCredentialUri' => 'https://oauth2.googleapis.com/token',
            'scope' => 'https://www.googleapis.com/auth/firebase.messaging',
        ]);
        $token = $oauth->fetchAuthToken();
        if (!isset($token['access_token'])) throw new Exception("Access token missing");
    } catch (Exception $e) {
        error_log("Auth error: " . $e->getMessage());
        return false;
    }

    $notificationTitle = 'Message from ' . ($messageRow['sender_username'] ?? 'User');
    $notificationBody = substr($messageRow['message'], 0, 100) . '...';
    $timestamp = strtotime($messageRow['created_at'] ?? 'now');

    $fcmMessage = [
        'message' => [
            'token' => $messageRow['receiver_fcm_token'],
            'notification' => [
                'title' => $notificationTitle,
                'body' => $notificationBody,
            ],
            'data' => [
                'click_action' => 'FLUTTER_NOTIFICATION_CLICK',
                'screen' => 'chat',
                'message_id' => (string)($messageRow['id']),
                'sender_id' => (string)($messageRow['sender_id']),
                'receiver_id' => (string)($messageRow['receiver_id']),
                'sender_username' => $messageRow['sender_username'] ?? '',
                'reels_video' => $messageRow['sender_reels_video'] ?? '',
                'message_content' => $messageRow['message'] ?? '',
                'image_url' => $messageRow['image_url'] ?? '',
                'is_location' => (string)($messageRow['is_location'] ?? '0'),
                'latitude' => (string)($messageRow['latitude'] ?? ''),
                'longitude' => (string)($messageRow['longitude'] ?? ''),
                'timestamp' => (string)$timestamp
            ],
            'android' => [
                'priority' => 'high',
                'notification' => [
                    'channel_id' => 'high_importance_channel',
                    'sound' => 'default'
                ]
            ],
            'apns' => [
                'headers' => [
                    'apns-priority' => '10'
                ],
                'payload' => [
                    'aps' => [
                        'sound' => 'default',
                        'badge' => 1,
                        'mutable-content' => 1
                    ]
                ]
            ]
        ]
    ];

    try {
        $client = new Client();
        $response = $client->post("https://fcm.googleapis.com/v1/projects/{$projectId}/messages:send", [
            'headers' => [
                'Authorization' => 'Bearer ' . $token['access_token'],
                'Content-Type' => 'application/json',
            ],
            'json' => $fcmMessage,
            'timeout' => 15,
        ]);

        $pdo->prepare("UPDATE messages SET is_notified = 1 WHERE id = ?")
            ->execute([$messageId]);

        error_log("✅ FCM Sent for message ID $messageId");
        return true;
    } catch (RequestException $e) {
        $response = $e->getResponse();
        $error = $response ? $response->getBody()->getContents() : $e->getMessage();
        error_log("❌ FCM failed for ID $messageId: $error");
        return false;
    }
}

// ----------------------- AUTO SCAN FOR NEW MESSAGES -----------------------
try {
    $pdo = new PDO("mysql:host=localhost;dbname=dreams", "root", "");
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // Get all messages that are not yet notified
    $stmt = $pdo->query("SELECT id FROM messages WHERE is_notified = 0 ORDER BY created_at ASC LIMIT 10");
    $messages = $stmt->fetchAll(PDO::FETCH_COLUMN);

    foreach ($messages as $msgId) {
        sendNotificationForMessage($msgId);
    }
} catch (PDOException $e) {
    error_log("DB Error: " . $e->getMessage());
}

// ----------------------- MANUAL TESTING OPTION -----------------------
if (isset($_GET['test_message'])) {
    header('Content-Type: text/plain');
    $messageId = (int)$_GET['test_message'];
    echo sendNotificationForMessage($messageId) ? "SUCCESS" : "FAILED";
    exit;
}
